; Weather information for Savannah, GA

globals [ 
  month 
  year 
  av-hi-temp 
  av-low-temp 
  rel-density-adjust
  fire-spread
  HA-per-cell
  juvenile-death-prob-per-month
  adult-death-prob-per-month
  senior-death-prob-per-month
  movie-recording?
]
patches-own [ 
  pct-herbs 
  pct-woody 
  burn-potential 
  burning? 
  max-GT
  GT-space-available
  eggs-juveniles
  burrows
  soil
  trails
  tmp
]

breed [ GT a-GT ]

GT-own [
  carapace-length
  sex
  age
  energy-reserve
  move-motivation
  move-count
  disperser?
]

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; SETUP and GO
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to start-stop-movie
  ifelse movie-recording?
  [
    movie-close
    set movie-recording? FALSE
  ]
  [
    movie-start "GT.mov"
    movie-set-frame-rate 3
    set movie-recording? TRUE
  ]
end

to setup-random-habitat
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  ask patches [
    set pct-woody 1 + random-float 100
    let max-pct-herbs 100
    if pct-woody >= 80 [ set max-pct-herbs 5 * ( 100 - pct-woody ) ]
    set pct-herbs 1 + random-float max-pct-herbs
    set burning? false
  ]
  set HA-per-cell 1
  set-patch-colors
  set-location
  initialize-tortoises
  setup-plot
  do-plot
end

to setup-grid-habitat
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  ask patches [
    ifelse (( pxcor mod  spacing = 0) and (pycor mod  spacing = 0 ))
    [
      set pct-woody 0
      set pct-herbs 80
    ]
    [
      set pct-woody 100
      set pct-herbs 0
    ]
    set burning? false
  ]
  set HA-per-cell 1
  set-patch-colors
  set-location
  initialize-tortoises
  setup-plot
  do-plot
end

to setup-center-habitat
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  ask patches [
    ifelse distancexy 0 0 < radius 
    [
      set pct-woody 0
      set pct-herbs 80
    ]
    [
      set pct-woody 100
      set pct-herbs 0
    ]
    set burning? false
  ]
  set HA-per-cell 1
  set-patch-colors
  set-location
  initialize-tortoises
  setup-plot
  do-plot
end

to setup-Benning
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  read-map "maps/burrows"  ask patches [ set burrows tmp ]
  read-map "maps/herb"     ask patches [ set pct-herbs tmp ]
  read-map "maps/soil"     ask patches [ set soil tmp ]
  read-map "maps/trails"   ask patches [ set trails tmp ]
  read-map "maps/woody"    ask patches [ set pct-woody tmp ]
  set HA-per-cell (.6 * .6)
  set-patch-colors
  set-location
  initialize-tortoises-Benning
  setup-plot
  do-plot
end

to read-map [ map-name ]
  file-open map-name
  let line file-read-line  print line
  set line file-read-line  print line
  set line file-read-line  print line
  set line file-read-line  print line
  set line file-read-characters 6
  let map-rows file-read   print map-rows
  set line file-read-characters 6
  let map-cols file-read   print map-cols
  let map-row 0
  let row max-pycor
  let val 0
  while [ map-row < map-rows ]
  [
    let map-col 0
    let col min-pxcor
    while [ map-col < map-cols ]
    [
      set val file-read
      if (row >= min-pycor) AND (col <= max-pxcor)
      [
        ask patch col row [ set tmp val ]
      ]
      set map-col map-col + 1
      set col col + 1
    ]
    set map-row map-row + 1
    set row row - 1
  ]
  file-close
end

to set-default-values
  set juvenile-CL 50
  set max-GT-per-HA 3
  set egg-to-juvenile-survival 5
  set min-juvenile-age 1
  set min-adult-age  10
  set min-senior-age  40
  set senior-death-prob 20
  set juvenile-death-prob 5
  set woody-growth-rate 5
  set pct-woody-lost-to-fire 95
  set herb-growth-rate 10
  set pct-herbs-lost-to-fire 10
  set lightening .01
  set burn-probability 20
  set carapace-rate 20
  set max-CL-female 330
  set max-CL-male 310
  set dispersers 2
end

to set-location
  set month 1
  set year 1
; Fort Benning Area 
  set av-low-temp        [ 37  39  46  52  61  69  72  71  66  33  46  39 ] ;; average low temp by month
  set av-hi-temp         [ 57  62  69  77  83  90  92  91  86  77  68  59 ] ;; average high temp by month
  set rel-density-adjust [ 1.0 1.0 1.4 3.0 3.0 3.0 2.0 1.8 1.2 1.0 1.0 1.0 ] ;;
; Savannah Area
;  set av-low-temp    [ 38.0 40.9 47.5 52.9 61.3 68.1 71.8 71.3 67.3 56.1 46.9 40.1 ] ;; average low temp by month
;  set av-hi-temp     [ 60.4 64.1 71.0 77.7 84.3 89.5 92.3 90.3 86.0 78.1 70.5 62.6 ] ;; average high temp by month
;  set rel-food-avail [ .3   .3   .5   1.   1.   1.   .8   .6   .4   .3   .3   .3   ] ;; 
end

to initialize-tortoises  
  ct
  set-default-shape GT "turtle"
  create-GT initial-tortoises  ;; create the tortoises, then initialize their variables
  [
    let startage 1 + random 50
    initialize-GT startage
    let ok 0
    while [ ok = 0 ]
    [
      let tempx random-xcor
      let tempy random-ycor
      if ( [ pct-herbs] of patch-at tempx tempy > 30 ) [
        setxy tempx tempy
        set ok 1
      ]
    ]
  ]
end

to initialize-tortoises-Benning  
  ct
  set-default-shape GT "turtle"
  ask patches with [ burrows > 0 ]
  [ sprout-GT burrows
    [
      let startage 1 + random 50
      initialize-GT startage
    ]
  ]
end

to initialize-GT [ start-age ]
  set age start-age
  ifelse random 2 = 0 [ set sex "F" set color 138 ] [ set sex "M" set color 98 ]
  set label-color blue - 2
  set move-motivation 0
  set energy-reserve 5
     
  ; set and grow carapace
  set carapace-length juvenile-CL
  let yr 1
  while [ yr < age ] [ 
    grow-carapace
    set yr yr + 1
  ]

  set size .5 + ( carapace-length / max-CL-female )
  set label-color blue - 2
  set move-count 0
  set disperser? 0
end

to go
  set month month + 1
  if month > 12 [ set month 1 set year year + 1 ]
  if ( month = 5 ) and (veg-growth = true) [ grow-vegetation ]  ; vegetation is allowed to grow once a year - May
  if ( month = 5 ) and (fire?) [ do-ground-fires ]              ; vegetation is allowed to burn once a year - May
    set-patch-colors

  if ( run-tortoise ) [
    without-interruption [
    ask patches [ 
      calculate-max-GT
      process-eggs-hatchlings
      move
    ]]
    ; convert annual death rates to monthly death rates
    ; calculated here in case user adjusts the rates during a run
;    set juvenile-death-prob-per-month 100 * ( 1 - e ^ (ln (1 - (juvenile-death-prob / 100)) / 12))
;    set adult-death-prob-per-month 100 * ( 1 - e ^ (ln (1 - (adult-death-prob / 100)) / 12))
;    set senior-death-prob-per-month 100 * (1 - e ^ (ln (1 - (senior-death-prob / 100)) / 12))
    set juvenile-death-prob-per-month 100 * ( ( ( 1 + ( juvenile-death-prob / 100 ) ) ^ ( 1 / 12 ) ) - 1 )
    set adult-death-prob-per-month    100 * ( ( 1 + ( adult-death-prob / 100 ) ) ^ ( 1 / 12 ) - 1 )
    set senior-death-prob-per-month   100 * ( ( 1 + ( senior-death-prob / 100 ) ) ^ ( 1 / 12 ) - 1 )
    print juvenile-death-prob
    print juvenile-death-prob-per-month
    ask GT [ 
      disperse
      grow
      eat
      reproduce
      potentially-die
     ]
  ]
  do-plot
  if movie-recording? [ movie-grab-interface ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; TORTOISE PROCEDURES
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to calculate-max-GT
  ifelse pct-herbs < 20
  [
    set max-GT 0
  ]
  [
    let adj-pct-herbs pct-herbs / 80
    if adj-pct-herbs > 1.0 [ set adj-pct-herbs 1.0 ]
    set max-GT adj-pct-herbs * max-GT-per-HA * ( item (month - 1) rel-density-adjust ) 
  ]
  set GT-space-available ( HA-per-cell * max-GT ) - count GT-here
  if training [ set GT-space-available GT-space-available * ( 100 - trails ) / 100 ]
end

to move
  ;; identify motivation to move
  ;  Tortoises will remain in burrows and therefore in their 1 ha patch when the av low is less than 50 degrees F
  if ( item ( month - 1 ) av-low-temp > 50 )
  [
    while [ count GT-here > max-GT ] [  
      ask min-one-of GT-here [ carapace-length ] [ move-from-patch ]
    ]
  ]
end

to disperse
  ;; a small % of tortoises are dispersers each year
  ;  Tortoises will remain in burrows and therefore in their 1 ha patch when the av low is less than 50 degrees F
  if ( item ( month - 1 ) av-low-temp > 50 and disperser? = 1 ) [
    move-from-patch
  ]
end

to move-from-patch
  let winner max-one-of neighbors [GT-space-available]
  let original-patch patch-here
  setxy ([pxcor] of winner) ([pycor] of winner)
  ask winner [ calculate-max-GT ]
  ask original-patch [ calculate-max-GT ]
  set energy-reserve energy-reserve - 1
  set move-count move-count + 1
end

to eat
  if ( item ( month - 1 ) av-low-temp > 50 )
  [
    set energy-reserve energy-reserve + .5
    if ( energy-reserve > 5 ) [ set energy-reserve 5 ]
  ]
end

to grow
  set age age + ( 1 / 12 )
  if round month = 1
  [
    ifelse random 100 < dispersers [ set disperser? 1 ] [ set disperser? 0 ]
    report-moves
    set move-count 0
    grow-carapace
    set size .5 + ( carapace-length / max-CL-female )
  ]
end

to grow-carapace
  let max-CL max-CL-female
  if ( sex = "M" ) [ set max-CL max-CL-male ]
  set carapace-length carapace-length + carapace-rate * ( 1 - carapace-length / max-CL )
end

to reproduce 
  if ( sex = "F" and month = 4 and age > min-adult-age and age < min-senior-age and female-reproduction-probability > random 100 )
  [
    set eggs-juveniles eggs-juveniles + eggs-per-female
  ]
end

to potentially-die
  if energy-reserve < 0 [ report-moves die]
  ifelse ( age < min-adult-age )
    [ if ( random-float 100 < juvenile-death-prob-per-month ) [ report-moves die ] ]
    [ ifelse ( age < min-senior-age )
      [ if ( random-float 100 < adult-death-prob-per-month ) [ report-moves die ] ]
      [ if ( random-float 100 < senior-death-prob-per-month ) [ report-moves die ] ]
    ]
end

to report-moves
;  file-open "GT-movement.txt"
;  file-show  round age + " " + sex + " " + move-count
;  file-close
end

to process-eggs-hatchlings
  if ( month = 4 ) [
    while [ eggs-juveniles > 0 ] [
      if ( random-float 100 < egg-to-juvenile-survival ) [
        sprout-GT 1 [ initialize-GT 1 ]
      ]
    set eggs-juveniles eggs-juveniles - 1
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; VEGETATION PROCEDURES
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to set-patch-colors
  ask patches [ set pcolor approximate-rgb ( 256 * (pct-woody / 100)) (256 * (pct-herbs / 100)) 0 ]
end

to do-ground-fires
  ask patches with [ random-float 1 < lightening ] [ set burning? true set pcolor black]  ; start fires
  while [ any? patches with [ burning? = true ] ] [                                       ; burn fires
    ask patches with [burning? = true] [
      set burn-potential  burn-probability * ( pct-herbs + pct-woody ) / 10000 ;
      if ( random-float 1.0 < burn-potential ) [
        set pct-woody ( pct-woody * ( 100 - pct-woody-lost-to-fire ) / 100 )
        set pct-herbs ( pct-herbs * ( 100 - pct-herbs-lost-to-fire ) / 100 )
        if pct-herbs < 10 [ set pct-herbs 10 ]
        without-interruption [ ask neighbors [ set burning? true set pcolor black ] ]
      ]
      set burning? false
    ]
  ]
end

to grow-vegetation  ; happens once a year
  ask patches [
    set pct-woody pct-woody + (woody-growth-rate / 100)  * ( 100 - pct-woody )
    set pct-herbs pct-herbs + (herb-growth-rate / 100)  * ( 100 - pct-herbs )
    let max-pct-herbs 100
    if pct-woody >= 80 [ set max-pct-herbs 5 * ( 100 - pct-woody ) ]
    if ( pct-herbs > max-pct-herbs )  [ set pct-herbs max-pct-herbs ]                     ; knock herbs-grass back
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; PLOTTING PROCEDURES
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-plot
  set-current-plot "Average % Cover"
  set-current-plot-pen "herbs" plot ( mean [ pct-herbs ] of patches )
  set-current-plot-pen "woody" plot ( mean [ pct-woody ] of patches )

  set-current-plot "Tortoise Count"
  set-current-plot-pen "default" plot ( count GT )

  set-current-plot "Age Distribution"
  histogram [ age ] of GT          ; using the default plot pen
end

to setup-plot ;; set up plotting
  set-current-plot "Average % Cover"
  set-plot-y-range 0 100

  set-current-plot "Age Distribution"
  set-plot-x-range 0 60
  set-plot-y-range 0 count GT
  set-histogram-num-bars 10
  set movie-recording? FALSE
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; REPORTING PROCEDURES
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


    
@#$#@#$#@
GRAPHICS-WINDOW
384
10
944
491
27
22
10.0
1
10
1
1
1
0
0
0
1
-27
27
-22
22
0
0
1
ticks
30.0

BUTTON
5
32
225
65
NIL
setup-random-habitat
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
240
31
372
64
Run Continuous
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
582
548
853
703
Average % Cover
Time
Percent Cover
0.0
100.0
0.0
100.0
true
true
"" ""
PENS
"herbs" 1.0 0 -10899396 true "" ""
"woody" 1.0 0 -2674135 true "" ""

MONITOR
736
709
786
754
month
month
1
1
11

MONITOR
791
709
848
754
year
year
1
1
11

SLIDER
203
707
390
740
burn-probability
burn-probability
0
100
20
1
1
%
HORIZONTAL

SLIDER
8
715
179
748
woody-growth-rate
woody-growth-rate
0
100
5
1
1
%/yr
HORIZONTAL

SLIDER
7
752
180
785
herb-growth-rate
herb-growth-rate
0
100
10
1
1
%/yr
HORIZONTAL

SLIDER
203
670
390
703
lightening
lightening
0
0.02
0.01
1.0E-4
1
strikes/yr/ha
HORIZONTAL

SWITCH
8
679
123
712
veg-growth
veg-growth
1
1
-1000

SWITCH
263
559
378
592
run-tortoise
run-tortoise
0
1
-1000

SLIDER
189
311
368
344
initial-tortoises
initial-tortoises
0
1000
100
1
1
NIL
HORIZONTAL

MONITOR
582
709
686
754
Num Tortoises
count GT
1
1
11

SLIDER
6
458
168
491
min-adult-age
min-adult-age
0
20
10
1
1
years
HORIZONTAL

SLIDER
189
444
378
477
juvenile-death-prob
juvenile-death-prob
0
10
5
0.1
1
%/yr
HORIZONTAL

SLIDER
6
495
170
528
min-senior-age
min-senior-age
30
100
40
1
1
years
HORIZONTAL

SLIDER
190
518
379
551
senior-death-prob
senior-death-prob
0
100
20
1
1
%/yr
HORIZONTAL

SLIDER
189
407
377
440
egg-to-juvenile-survival
egg-to-juvenile-survival
0
10
5
1
1
%
HORIZONTAL

SLIDER
189
275
367
308
max-GT-per-HA
max-GT-per-HA
1
5
3
0.1
1
GT
HORIZONTAL

SLIDER
203
743
391
776
pct-herbs-lost-to-fire
pct-herbs-lost-to-fire
0
99
10
1
1
%
HORIZONTAL

SLIDER
204
780
391
813
pct-woody-lost-to-fire
pct-woody-lost-to-fire
0
99
95
1
1
%
HORIZONTAL

PLOT
406
549
566
669
Tortoise Count
Time
Count
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" ""

PLOT
405
674
565
813
Age Distribution
Count
Age Cohort
0.0
60.0
0.0
10.0
true
false
"" ""
PENS
"default" 10.0 1 -16777216 true "" ""

BUTTON
240
68
372
101
Run 1 month
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
3
274
165
307
juvenile-CL
juvenile-CL
30
100
50
1
1
mm
HORIZONTAL

SLIDER
4
312
165
345
carapace-rate
carapace-rate
1
40
20
1
1
mm/yr
HORIZONTAL

SLIDER
5
349
166
382
max-CL-female
max-CL-female
300
500
330
10
1
mm
HORIZONTAL

BUTTON
124
220
270
253
NIL
set-default-values
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
6
420
167
453
min-juvenile-age
min-juvenile-age
1
5
1
1
1
years
HORIZONTAL

SLIDER
189
481
379
514
adult-death-prob
adult-death-prob
0
10
3
1
1
%/yr
HORIZONTAL

SLIDER
5
598
189
631
eggs-per-female
eggs-per-female
1
20
6
1
1
eggs
HORIZONTAL

SLIDER
5
560
248
593
female-reproduction-probability
female-reproduction-probability
20
100
60
1
1
%/yr
HORIZONTAL

SWITCH
204
634
319
667
fire?
fire?
1
1
-1000

BUTTON
96
68
225
101
NIL
setup-center-habitat
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
3
68
95
101
radius
radius
1
20
5
1
1
NIL
HORIZONTAL

SLIDER
5
384
167
417
max-CL-male
max-CL-male
300
350
310
1
1
mm
HORIZONTAL

SLIDER
189
347
368
380
dispersers
dispersers
0
10
2
1
1
%
HORIZONTAL

TEXTBOX
264
614
300
632
FIRE
11
0.0
0

TEXTBOX
51
542
148
560
REPRODUCTION
11
0.0
0

TEXTBOX
53
255
104
273
GROWTH
11
0.0
0

TEXTBOX
253
389
304
407
DEATH
11
0.0
0

TEXTBOX
59
660
134
678
VEGETATION
11
0.0
0

TEXTBOX
239
257
315
275
POPULATION
11
0.0
0

TEXTBOX
8
14
158
32
Step 1: Select Habitat
11
0.0
0

TEXTBOX
230
13
380
31
Step 3: Run
11
0.0
0

TEXTBOX
7
231
118
249
Step 2: Set Variables
11
0.0
0

BUTTON
97
104
224
137
NIL
setup-grid-habitat
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
4
103
96
136
spacing
spacing
2
10
3
1
1
NIL
HORIZONTAL

BUTTON
98
141
221
174
NIL
setup-Benning
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
233
144
338
177
training
training
0
1
-1000

BUTTON
607
780
746
813
NIL
start-stop-movie
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## GOPHER TORTOISE HABITAT SIMULATION

This project explores the implications of the ability to use field studies of the Gopher Tortoise to project the implications of alternative land management opportunities on the population viability of populations.

This version grows forbs-grass, which is food for the GT and is fire resistant.

## HOW TO RUN

1) Choose a habitat.   There are three options:

setup-random-habitat - Each patch is randomly assigned a percentage cover of herbaceous and woody vegetation and the chosen number of tortoises are distributed in acceptable habitat (> 20 herbaceous)

setup-center-habitat - A large patch of perfect habitat is placed in the center (80% herbaceous and 0% woody) and tortoises are distrubted within it. The radius of the patch is determined by the associated "radius" variable.

setup-grid-habitat - A regular patchwork of 1-ha good habitat (80% herbaceous) is placed on a background of bad habitat (100% woody).  Tortoise are distributed on the good habitat.

2) Set variables.

They can be set in two ways.  Hitting the "set-default-values" button will reset all of the variables to something that will give you reasonable results.  You may also use the various sliders to set individual variables.

3) Run the model

Hit the "Run Continuous" toggle button to run the model until you hit this button again, or his the "Run 1 month" button to do just that.

## KEY ASPECTS

Tortoises grow, move, reproduce, and die based on the different variables and on the conditions of their surroundings (vegetation and other tortoises).

Vegetation grows at the rates set.  Generally, herbaceous grows more rapidly, but woody eventually takes over.

Fire is a random event that burns across the landscape to maintain a landscape patchwork within which a tortoise population must survive.

## NEXT DEVELOPMENT STEPS

Track genetic flow  
Manage land with prescribed burns  
Import real landscapes  
Accommodate different elevations  
Accommodate different soil types (ease of digging)

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

rabbit
false
0
Circle -7500403 true true 76 150 148
Polygon -7500403 true true 176 164 222 113 238 56 230 0 193 38 176 91
Polygon -7500403 true true 124 164 78 113 62 56 70 0 107 38 124 91

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="100-year" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-Benning</setup>
    <go>go</go>
    <final>file-open "results"
ask GT [ file-type who file-type " " file-type xcor file-type " " file-type ycor file-type " " file-type sex file-type " " file-print age ]
file-print ""
file-close</final>
    <timeLimit steps="1200"/>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="carapace-rate">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pct-woody-lost-to-fire">
      <value value="95"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-GT-per-HA">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="run-tortoise">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-juvenile-age">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-CL-female">
      <value value="330"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-number-tortoises">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="burn-probability">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fire">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-senior-age">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="veg-growth">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spacing">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="herb-growth-rate">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-adult-age">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="training">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-survival">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-death-prob">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-death-prob">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="senior-death-prob">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-CL-male">
      <value value="310"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="radius">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pct-herbs-lost-to-fire">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-CL">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lightening">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="woody-growth-rate">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migrators">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
